# -*- coding:utf-8 -*-
from defusedxml import ElementTree as ET
import os
from common.devNode import DevNode
from common import commonFunction
from common.commonFunction import getResource
from common.constant import SCRIPT_RET_CODE
from common import cliCmdManager
from common.cTV2R2 import cHandleTypeList
from common import utils

def execute(context):
    '''
    @summary: Initialize device node.
    @date: 2014-07-28
    @param context: context object
    @return: 
    '''
    logger = context.get('logger')
    logger.info('[prepare] start to prepare handle...')
    resource = getResource(context)
    devNode = context.get('devNode')
    
    if not devNode:
        devNode = DevNode(context)
        context['sshConnector'] = devNode.getSshConnector(context)
        context['devNode'] = devNode
        logger.info(str(devNode))
        
        if not isSshConnectionNormal(context):
            errMsg = resource.getString('err.create.SSH.connection.failed')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        if not devNode.isSystemNormal(context):
            errMsg = resource.getString('err.sys.status.abnormal')
            if devNode.deviceName.find('9500') >= 0:
                errMsg = errMsg.replace('9000', '9500')
            if devNode.deviceName.find('9000-K') >= 0:
                errMsg = errMsg.replace('9000', '9000-K')
            return (SCRIPT_RET_CODE.FAIL, errMsg)

    errCode, errMsg = parseStdCfgFile(context)
    logger.info('[prepare] finish to prepare handle...')
    
    return (errCode, errMsg)

def isSshConnectionNormal(context):
    '''
    @summary: Check if the SSH connection is normal
    @date: 2014-08-18
    @param context: context object
    @return: 
    '''
    logger = context.get('logger')
    sshConnector = context['sshConnector']
    if not sshConnector.getConnectionNoException():
        logger.error('Get SSH connection failed!')
        return False
    else:
        logger.info('Get SSH connection successfully!')
        return True
    
def parseStdCfgFile(context):
    '''
    @summary: load and parse configure file, then load to memory
    @date: 2014-07-28
    @param context: context object
    @param xmlFile: the absolute path of configure file
    @param contextKey: the key of context map object
    @return: 
    '''
    logger = context.get('logger')
    resource = getResource(context)
    
    cfgFile4Standard = context.get('path') + os.sep + 'script' + os.sep + 'config' + os.sep + 'cmd4IBMS.xml'
    root = ET.parse(cfgFile4Standard)
    itemList = root.findall('item')
    itemCmdListDict = {}
    errMsg = ''
    errCode = SCRIPT_RET_CODE.SUCCESS
    
    try:
        for item in itemList:
            itemName = item.attrib['name']
            itemCmdList = []
            baseCmd = item.findall('cmd')[0].attrib['command']
            if not item.findall('cmd/param'):
                fullCmd = baseCmd
                itemCmdList.append(fullCmd)
            else:
                paraName = item.findall('cmd/param')[0].attrib['paraName']
                paraValCmd = item.findall('cmd/param')[0].attrib['LoopBaseCmd']
                filedName = item.findall('cmd/param')[0].attrib['loop']
                for paraVal in getParaValByCLI(context, paraValCmd, filedName):
                    fullCmd = baseCmd + ' ' + paraName + '=' + paraVal
                    itemCmdList.append(fullCmd)
            itemCmdListDict[itemName] = itemCmdList
            componentNumKey = itemName + 'Num'
            if 'PowerModule' == itemName:
                context[componentNumKey] = getPowerModuleNum(context, baseCmd)
            else:
                context[componentNumKey] = len(itemCmdList)
            logger.info(componentNumKey + ':' + str(context[componentNumKey]))
    except:
        logger.error('Parse IBMS configure file caught exception.')
        errMsg = resource.getString('err.parse.IBMS.cfg.failed')
        errCode = SCRIPT_RET_CODE.FAIL
    else:
        logger.info('Standard item name-cmdList:' + str(itemCmdListDict))
        context['stdItemDict'] = itemCmdListDict
        errCode = SCRIPT_RET_CODE.SUCCESS
    finally:
        return (errCode, errMsg)
    
    
def getPowerModuleNum(context, cmd):
    '''
    @summary: Get the number of power module
    @date: 2014-08-22
    @param context: context object
    @param cmd: the command of power module
    @return: the number of power module
    '''
    
    sshConnector = context['sshConnector']
    logger = context.get('logger')
    
    dev = context.get("dev")
    userName = dev.getLoginUser().getUserName()
    userPasswd = dev.getLoginUser().getPassword()

    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger, userName, userPasswd)
    if not isSucc:
        logger.error('[prepare] sExecute command [' + cmd + '] failed!')
        return 0
    cmdRetLineList = cmdRet.splitlines()
    isSwitchOn = False
    counter = 0
    for line in cmdRetLineList:
        if -1 != line.find('---'):
            isSwitchOn = True
            continue
        if -1 != line.find(':/>'):
            isSwitchOn = False
        if not line.strip():
            continue
        if isSwitchOn:
            counter += 1
    logger.info('PowerModuleNum=' + str(counter))
    return counter
    
    
def getParaValByCLI(context, cmd, fieldName):
    '''
    @summary: Get the value of parameter.
    @date: 2014-07-28
    @param context: context object
    @param cmd: command
    @param fieldName: 
    @return: the value list of parameter
    '''
     
    sshConnector = context['sshConnector']
    logger = context.get('logger')
    dev = context.get('dev')
    userName = dev.getLoginUser().getUserName()
    userPasswd = dev.getLoginUser().getPassword()

    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger, userName, userPasswd)
    if not isSucc:
        logger.error('[prepare] Execute base loop command [' + cmd + '] failed!')
        return []
    
    logger.info('filedName:' + str(fieldName))
    if not utils.checkCliInfoValid(cmdRet):
        logger.warn('[prepare] Command execute result is invalid!')
        return []
    formatFunction = cHandleTypeList(cmdRet, [fieldName])
    fieldInfoDictList = formatFunction.handle()[1]
    fieldValList = [fieldInfoDict.get(fieldName, '') for fieldInfoDict in fieldInfoDictList]
    if '' in fieldValList:
        fieldValList.remove('')
    
    logger.info('FieldValueList' + str(fieldValList))
    return fieldValList
